DOUBLE PRECISION FUNCTION corponegro(comp_onda)
    !Função de corpo negro para um dado comprimento de onda (comp_onda) e temperatura T = 2500K
    
    IMPLICIT NONE
    
    DOUBLE PRECISION, INTENT(IN) :: comp_onda !Comprimento de onda a ser calculada a função
    DOUBLE PRECISION :: expoente, divisor, k, h, c, pi !Partes do cálculo da função e constantes
    
    !Constantes em CGS (constante de boltzmann, de planck e da velocidade da luz no vácuo) e uma aproximação para pi
    k = 1.381D-16
    h = 6.626D-27
    c = 2.998D10
    pi = 3.142
                        
    !Calculando a função
    expoente = (h * c / (comp_onda * k * 2500.)) !2500K = T
    divisor = comp_onda**5. * (exp(expoente) - 1.)
        
    corponegro = (2. * h * c**2.) / divisor
    
END FUNCTION corponegro

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

DOUBLE PRECISION FUNCTION simpson(x0, x1, n)
    !Calcula a integral da função de corpo negro em um intervalo [x0, x1] dividido em n subintervalos

    IMPLICIT NONE
    
    DOUBLE PRECISION, INTENT(IN) :: x0, x1 !Limites do intervalo
    DOUBLE PRECISION, SAVE :: anterior !Estimativa anterior calculada (salva a cada chamada da função)
    DOUBLE PRECISION, EXTERNAL :: corponegro !Função a ser integrada
    DOUBLE PRECISION :: somatoria, termo, h, trapezio !Variáveis auxiliares, tamanho h dos subintervalos e valor dado pelo trapezio
    INTEGER :: n !Quantidade n de subintervalos
    
    !Definindo h, termo e valor inicial da somatoria
    h = (x1 - x0) / n
    termo = x0 + h
    somatoria = 0.
    
    !Caso haja apenas 1 intervalo
    IF (n .EQ. 2) THEN
    
        anterior = (h * 2) * (corponegro(x0) + corponegro(x1)) / 2
        
        DO WHILE (termo < x1)
        
            somatoria = somatoria + corponegro(termo)            
            termo = termo + 2 * h
            
        END DO
        
        !Cálculo da integral por trapézios
        trapezio = anterior / 2 + h * somatoria
       
    !No caso de mais intervalos
    ELSE
    
        DO WHILE (termo < x1)
        
            somatoria = somatoria + corponegro(termo)            
            termo = termo + 2 * h
            
        END DO
        
        !Cálculo da integral por trapézios
        trapezio = anterior / 2 + h * somatoria
        
    END IF
    
    !Calculando a integral por Simpson
    simpson = (4 * trapezio - anterior) / 3
    
    !Atualizando o valor a ser salvo
    anterior = trapezio
    
END FUNCTION simpson

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

PROGRAM integracao_simpson

    IMPLICIT NONE

    DOUBLE PRECISION, EXTERNAL :: simpson !Função que realiza a integração
    DOUBLE PRECISION, DIMENSION(7) :: eps !Array com as precisões desejadas
    DOUBLE PRECISION :: lbd0, lbd1, eps_integral, integral0, integral1 !Extremos do intervalo de comprimento de onda, erro relativo de cada passo, estimativas anterior e atual da integral
    INTEGER :: n, i !Número n de intervalos, variável auxiliar
    
    !Definindo o intervalo
    lbd0 = 5.D-5 !0.5 microns -> cm
    lbd1 = 1.D-3 !10 microns -> cm
    
    !Definindo o array eps
    eps = (/1.D-4, 1.D-5, 1.D-6, 1.D-7, 1.D-8, 1.D-9, 1.D-10/)
    
    !Calculando as integrais para cada valor de eps
    n = 2
    integral0 = 0.
    integral1 = simpson(lbd0, lbd1, n)
    eps_integral = ABS((integral1 - integral0) / integral1)
    
    PRINT *, "As colunas abaixo correspondem, respectivamente a: epsilon, número de passos e valor da integral"
    
    DO i = 1, 7
    
        DO WHILE (eps_integral >= eps(i))

            n = n * 2
            integral0 = integral1
            integral1 = simpson(lbd0, lbd1, n)
            eps_integral = ABS((integral1 - integral0) / integral1)
        
        END DO

        PRINT *, eps(i), n, integral1
    
    END DO
    
END PROGRAM integracao_simpson
